<?php
namespace ZS\Usaepay\Model;
use Magento\Payment\Model\InfoInterface;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\LocalizedException;
use ZS\Usaepay\Helper\Config;
use ZS\Usaepay\Helper\Data;
use ZS\Usaepay\Model\Card;
use ZS\Usaepay\Model\CardFactory;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Quote\Api\Data\PaymentInterface;

class CCPaymentAction extends \Magento\Payment\Model\Method\Cc
{

    const METHOD_CODE 	= 'zs_usaepay';
	const CC_TOKEN 		= 'cc_token';
    const CARD_ID 		= 'card_id';
    const SAVE_CARD 	= 'save_card';

    protected $_code = self::METHOD_CODE;
    protected $_isGateway = true;
    protected $_canAuthorize = true;
    protected $_canCapture = true;
    protected $_canCapturePartial = true;
    protected $_canRefund = true;
    protected $_canRefundInvoicePartial = true;
    protected $_canVoid = true;
    protected $_canUseInternal = true;
    protected $_canUseCheckout = true;
    protected $_canSaveCc = false;
    protected $_authMode = 'auto';
    protected $loadedCardId;

    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Magento\Framework\Module\ModuleListInterface $moduleList,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $localeDate,
		Config $configHelper,
        Data $dataHelper,
		CardFactory $cardFactory,
        CustomerSession $customerSession,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $moduleList,
            $localeDate,
            $resource,
            $resourceCollection,
            $data
        );
		$this->configHelper 	= $configHelper;
		$this->cardFactory 		= $cardFactory;
		$this->dataHelper 		= $dataHelper;
		$this->customerSession 	= $customerSession;
//        $this->tran = $tranApi;
    }
	
	/**
     * @param DataObject $data
     * @return $this
     * @throws LocalizedException
     */
    public function assignData(DataObject $data)
    {
        parent::assignData($data);
        $additionalData = $data->getData(PaymentInterface::KEY_ADDITIONAL_DATA);
        if (!is_object($additionalData)) {
            $additionalData = new DataObject($additionalData ?: []);
        }
        $saveCard 		= $additionalData->getData(self::SAVE_CARD) ?: $data->getData(self::SAVE_CARD);
        $cardId 		= $additionalData->getData(self::CARD_ID) ?: $data->getData(self::CARD_ID);	
        $newAdditionalData = [
            self::SAVE_CARD 		=> $saveCard
        ];
        /** @var DataObject $info */
        $info = $this->getInfoInstance();
		if($data->getCcType() == 'GC') {
            $info->setCcType($data->getCcType())
                ->setCcOwner($data->getCcOwner())
                ->setCcLast4(substr($data->getCcNumber(), -4))
                ->setCcNumber($data->getCcNumber())
                ->setCcExpMonth(0)
                ->setCcExpYear(0)
                ->setCcCid(0)
                ->setCcUsaepayToken('');
            return $this;
        }
		$info->setAdditionalInformation($newAdditionalData);

        if ($cardId) {
            if($this->configHelper->forceCvvForSavedCards() && !$info->getData('cc_cid')) {
                throw new LocalizedException(__('CVV is required.'));
            }
            $card = $this->loadCard($cardId);
            /** @var DataObject $info */
            $info->addData([
                'cc_type' => $card->getData('cc_type'),
                'cc_number' => '************' . $card->getData('cc_last_4'),
                'cc_last_4' => $card->getData('cc_last_4'),
                'cc_exp_month' => (int)$this->dataHelper->getExpirationMonthFromCard($card),
                'cc_exp_year' => $this->dataHelper->getExpirationYearFromCard($card)
            ]);
			$info->setAdditionalInformation(self::CC_TOKEN, $card->getData('cc_token'));
            $this->loadedCardId = $cardId;
        }
        return $this;
    }

    public function validate()
    {
		$ccNumber = $this->getInfoInstance()->getCcNumber();
        if (preg_match('/^\*{11,12}\d{4}/', "", $ccNumber) === 0) {
            return $this;
        }
        /* if ($info->getCcType() != 'GC' && !$this->getConfigData('tokenization')) {
            return parent::validate();
        } */
        $errorMsg 			= false;
        $availableTypes 	= explode(',', $this->getConfigData('cctypes'));
        $ccNumber 			= $info->getCcNumber();
        $ccNumber 			= preg_replace('/[\-\s]+/', '', $ccNumber);
		$info->setCcNumber($ccNumber);
        if (in_array($info->getCcType(), $availableTypes)) {
            if ($info->getCcType() == 'GC') {
                if (!preg_match('/^[0-9]{19}$/', $info->getCcNumber())) {
                    $errorMsg = __('Gift card number mismatch with credit card type.');
                }
            } else {
                if (!$this->_validateExpDate($info->getCcExpYear(), $info->getCcExpMonth())) {
                    $errorMsg = __('Incorrect credit card expiration date.');
                }
            }
        } else {
            $errorMsg = __('Credit card type is not allowed for this payment method.');
        }
        if ($errorMsg) {
            throw new \Magento\Framework\Exception\LocalizedException($errorMsg);
        }
        return $this;
    }

    public function authorize(InfoInterface $payment, $amount)
    {
        // initialize transaction object
        $tran = $this->_initTransaction($payment);
        $useExtendedFraudProfiling = $this->getConfigData('extendedfraudprofiling');
        if ($useExtendedFraudProfiling) {
            $sessionId = $payment->getAdditionalInformation('usaepay_efpSessionId');
            if ($sessionId)
                $tran->session = $sessionId;
        }
        // general payment data
        $tran->cardholder = $payment->getCcOwner();
		$cardToken = null;
        if ($this->configHelper->isTokenizationEnabled()) {
            $cardToken = $this->getInfoInstance()->getAdditionalInformation(self::CC_TOKEN);
        }
		
        if ($payment->getCcType() != 'GC' && $cardToken) {
            $tran->card = $cardToken;
        } else {
            $tran->card = $payment->getCcNumber();
        }
        $tran->exp = sprintf('%02s', $payment->getCcExpMonth()) . substr($payment->getCcExpYear(), 2, 2);
        $tran->cvv2 = $payment->getCcCid();
        $tran->amount = $amount;
        $tran->ponum = $payment->getPoNumber();
        $tran->custreceipt = $this->getConfigData('custreceipt') == 0 ? false : true;
        $tran->custreceipt_template = trim($this->getConfigData('custreceipt_template'));
        // switch command based on pref
        if ($payment->getCcType() == 'GC') {
            $tran->command = 'giftcard:sale';
        } else {
            if ($this->getConfigData('payment_action') == self::ACTION_AUTHORIZE && $this->_authMode != 'capture')
                $tran->command = 'cc:authonly';
            else
                $tran->command = 'cc:sale';
        }

        // if order exists,  add order data
        $order = $payment->getOrder();
        if (!empty($order)) {

            $orderid = $order->getIncrementId();
            $tran->invoice = $orderid;
            $tran->orderid = $orderid;
            $tran->ip = $order->getRemoteIp();
            $tran->email = $order->getCustomerEmail();

            $tran->tax = $order->getTaxAmount();
            $tran->shipping = $order->getShippingAmount();

            $tran->description = ($this->getConfigData('description') ? str_replace('[orderid]', $orderid, $this->getConfigData('description')) : "Magento Order #" . $orderid);

            // billing info
            $billing = $order->getBillingAddress();
            if (!empty($billing)) {
                // avs data
                list($avsstreet) = $billing->getStreet();
                $tran->street = $avsstreet;
                $tran->zip = $billing->getPostcode();

                $tran->billfname = $billing->getFirstname();
                $tran->billlname = $billing->getLastname();
                $tran->billcompany = $billing->getCompany();
                $tran->billstreet = $billing->getStreetLine(1);
                $tran->billstreet2 = $billing->getStreetLine(2);
                $tran->billcity = $billing->getCity();
                $tran->billstate = $billing->getRegion();
                $tran->billzip = $billing->getPostcode();
                $tran->billcountry = $billing->getCountryId();
                $tran->billphone = $billing->getTelephone();
                $tran->custid = $billing->getCustomerId();
            }

            // shipping info
            $shipping = $order->getShippingAddress();
            if (!empty($shipping)) {
                $tran->shipfname = $shipping->getFirstname();
                $tran->shiplname = $shipping->getLastname();
                $tran->shipcompany = $shipping->getCompany();
                $tran->shipstreet = $shipping->getStreetLine(1);
                $tran->shipstreet2 = $shipping->getStreetLine(2);
                $tran->shipcity = $shipping->getCity();
                $tran->shipstate = $shipping->getRegion();
                $tran->shipzip = $shipping->getPostcode();
                $tran->shipcountry = $shipping->getCountryId();
                $tran->shipphone = $shipping->getTelephone();
            }
            // line item data
            foreach ($order->getAllVisibleItems() as $item) {
                if ($item->hasParentItem()) continue;
                $tran->addLine($item->getSku(), $item->getName(), '', $item->getPrice(), ($tran->command == 'cc:authonly' ? $item->getQtyToInvoice() : $item->getQtyInvoiced()), $item->getTaxAmount());
            }
        }

        if ($this->getConfigData('timeout') && is_numeric($this->getConfigData('timeout'))) {
            $tran->timeout = $this->getConfigData('timeout');
        }
		$info 				   = $this->getInfoInstance();
		$saveCard 			   = $info->getAdditionalInformation(self::SAVE_CARD);
		$isTokenizationEnabled = $this->configHelper->isTokenizationEnabled();
		if($saveCard && $isTokenizationEnabled){
			$tran->saveCard = true;
		}	
        // process transactions
        $tran->Process();
        
        // store response variables
        $payment->setCcApproval($tran->authcode)
                ->setCcTransId($tran->refnum)
                ->setCcAvsStatus($tran->avs_result_code)
                ->setCcCidStatus($tran->cvv2_result_code);

        if ($tran->resultcode == 'A') {
            if ($this->getConfigData('payment_action') == self::ACTION_AUTHORIZE)
                $payment->setLastTransId('0');
            else
                $payment->setLastTransId($tran->refnum);
			if($tran->saveCard){	
				$this->saveCard($tran->cardRef,$tran->cardType,$tran->maskedCardNum);	
			}
            $isResponseSet = false;
            if ($useExtendedFraudProfiling) {
                $useSuspectedFraudConfig = (int) $this->getConfigData('usesuspectedfraud');
                $isFraud = ($useSuspectedFraudConfig === 2 && $tran->profilerResponse == 'reject') ||
                        ($useSuspectedFraudConfig === 3 && ($tran->profilerResponse == 'reject' || $tran->profilerResponse == 'review'));

                if ($useSuspectedFraudConfig && $isFraud) {
                    $payment->setIsTransactionPending(true);
                    $payment->setIsFraudDetected(true);
                    $isResponseSet = true;
                }
            }

            if (!$isResponseSet) {
                $payment->setStatus(self::STATUS_APPROVED);
            }

            //ueLogDebug("CCPaymentAction::Authorize  Approved");
        } elseif ($tran->resultcode == 'D') {

            //ueLogDebug("CCPaymentAction::Authorize  Declined" );

            throw new LocalizedException(__('Payment authorization transaction has been declined:  ' . $tran->error));
        } else {

            //ueLogDebug("CCPaymentAction::Authorize  Error" );

            throw new LocalizedException(__('Payment authorization error:  ' . $tran->error . '(' . $tran->errorcode . ')'));
        }

        if ($useExtendedFraudProfiling && $tran->profilerResponse && !empty($order)) {
            $comment = "Extended Fraud Profiler Results:\n";
            if ($tran->profilerResponse)
                $comment .= "<br>response: {$tran->profilerResponse}\n";
            // score can be 0 so check it strictly against empty string
            if ($tran->profilerScore !== '')
                $comment .= "<br>score: {$tran->profilerScore}\n";
            if ($tran->profilerReason)
                $comment .= "<br>reason: {$tran->profilerReason}\n";
            $order->addStatusHistoryComment($comment);
            $order->save();
        }
		
    }
	
    public function quicksale(InfoInterface $payment, $amount)
    {
        // initialize transaction object
        $tran = $this->_initTransaction($payment);

        if (!$payment->getLastTransId())
            throw new LocalizedException(__('Unable to find previous transaction to reference'));

        // payment data
        $tran->refnum = $payment->getLastTransId();
        $tran->amount = $amount;
        $tran->ponum = $payment->getPoNumber();

        $tran->custreceipt = $this->getConfigData('custreceipt') == 0 ? false : true;
        $tran->custreceipt_template = trim($this->getConfigData('custreceipt_template'));

        // if order exists,  add order data
        $order = $payment->getOrder();
        if (!empty($order)) {

            $orderid = $order->getIncrementId();
            $tran->invoice = $orderid;
            $tran->orderid = $orderid;
            $tran->ip = $order->getRemoteIp();
            $tran->email = $order->getCustomerEmail();

            $tran->tax = $order->getTaxAmount();
            $tran->shipping = $order->getShippingAmount();

            $tran->description = ($this->getConfigData('description') ? str_replace('[orderid]', $orderid, $this->getConfigData('description')) : "Magento Order #" . $orderid);

            // billing info
            $billing = $order->getBillingAddress();
            if (!empty($billing)) {
                // avs data
                list($avsstreet) = $billing->getStreet();
                $tran->street = $avsstreet;
                $tran->zip = $billing->getPostcode();

                $tran->billfname = $billing->getFirstname();
                $tran->billlname = $billing->getLastname();
                $tran->billcompany = $billing->getCompany();
                $tran->billstreet = $billing->getStreetLine(1);
                $tran->billstreet2 = $billing->getStreetLine(2);
                $tran->billcity = $billing->getCity();
                $tran->billstate = $billing->getRegion();
                $tran->billzip = $billing->getPostcode();
                $tran->billcountry = $billing->getCountryId();
                $tran->billphone = $billing->getTelephone();
                $tran->custid = $billing->getCustomerId();
            }

            // shipping info
            $shipping = $order->getShippingAddress();
            if (!empty($shipping)) {
                $tran->shipfname = $shipping->getFirstname();
                $tran->shiplname = $shipping->getLastname();
                $tran->shipcompany = $shipping->getCompany();
                $tran->shipstreet = $shipping->getStreetLine(1);
                $tran->shipstreet2 = $shipping->getStreetLine(2);
                $tran->shipcity = $shipping->getCity();
                $tran->shipstate = $shipping->getRegion();
                $tran->shipzip = $shipping->getPostcode();
                $tran->shipcountry = $shipping->getCountryId();
                $tran->shipphone = $shipping->getTelephone();
            }
        }

        $tran->command = 'quicksale';

        // process transactions
        $tran->Process();

        // store response variables
        $payment->setCcApproval($tran->authcode)
                ->setCcTransId($tran->refnum)
                ->setCcAvsStatus($tran->avs_result_code)
                ->setCcCidStatus($tran->cvv2_result_code);

        if ($tran->resultcode == 'A') {
            if ($tran->refnum)
                $payment->setLastTransId($tran->refnum);
            $payment->setStatus(self::STATUS_APPROVED);
            //ueLogDebug("Transaction Approved");
        } elseif ($tran->resultcode == 'D') {
            //ueLogDebug("Transaction Declined");
            throw new LocalizedException(__('Payment authorization transaction has been declined:  ' . $tran->error));
        } else {
            //ueLogDebug("Transaction Error");
            throw new LocalizedException(__('Payment authorization error:  ' . $tran->error . '(' . $tran->errorcode . ')'));
        }

        return $this;
    }

    public function refund(InfoInterface $payment, $amount)
    {
        try {
            $error = false;

            $orderid = $payment->getOrder()->getIncrementId();

            list($avsstreet) = $payment->getOrder()->getBillingAddress()->getStreet();
            $avszip = $payment->getOrder()->getBillingAddress()->getPostcode();

            $tran = $this->_initTransaction($payment);

            if (!$payment->getLastTransId())
                throw new LocalizedException(__('Unable to find previous transaction to reference'));

            // payment data
            $tran->refnum = $payment->getLastTransId();
            $tran->amount = $amount;
            $tran->invoice = $orderid;
            $tran->orderid = $orderid;
            $tran->cardholder = $payment->getCcOwner();
            $tran->street = $avsstreet;
            $tran->zip = $avszip;
            $tran->description = "Online Order";
            $tran->cvv2 = $payment->getCcCid();
            $tran->command = 'quickcredit';

            if (!$tran->Process()) {
                $payment->setStatus(self::STATUS_ERROR);
                $error = __('Error in authorizing the payment: ' . $tran->error);
                throw new LocalizedException('Payment Declined: ' . $tran->error . ' (' . $tran->errorcode);
            } else {
                $payment->setStatus(self::STATUS_APPROVED);
            }

            if ($error !== false) {
                throw new LocalizedException($error);
            }
        } catch (\Exception $e) {
            $message = $e->getMessage();
            throw new LocalizedException(__('There was an error refunding the transaction: %1.', $message));
        }
        return $this;
    }

    public function capture(InfoInterface $payment, $amount)
    {
        try {
            if ($payment->getLastTransId() && $payment->getOrder()->getTotalPaid() > 0) {
                return $this->quicksale($payment, $amount);
            }

            //if we don't have a transid than we are need to authorize
            if (!$payment->getCcTransId() || $payment->getLastTransId()) {
                $this->_authMode = 'capture';
                return $this->authorize($payment, $amount);
            }

            $tran = $this->_initTransaction($payment);
            $tran->command = 'cc:capture';
            $tran->refnum = $payment->getCcTransId();

            $tran->amount = $amount;

            // process transaction
            $tran->Process();

            // look at result code
            if ($tran->resultcode == 'A') {
                $payment->setStatus(self::STATUS_APPROVED);
                $payment->setLastTransId($tran->refnum);
                return $this;
            } elseif ($tran->resultcode == 'D') {
                throw new LocalizedException(__('Payment authorization transaction has been declined:  ' . $tran->error));
            } else {
                throw new LocalizedException(__('Payment authorization error:  ' . $tran->error . '(' . $tran->errorcode . ')'));
            }
        } catch (\Exception $e) {
            $this->_logger->critical($e);
            throw new LocalizedException(__('There was an error capturing the transaction: %1.', $e->getMessage()));
        }
        return $this;
    }

    public function void(InfoInterface $payment)
    {
        if ($payment->getCcTransId()) {
            $tran = $this->_initTransaction($payment);
            $tran->command='creditvoid';
            $tran->refnum=$payment->getCcTransId();

            // process transactions
            $tran->Process();
            if($tran->resultcode=='A')
            {
               $payment->setStatus(self::STATUS_SUCCESS);

            } elseif($tran->resultcode == 'D') {
            	$payment->setStatus(self::STATUS_ERROR);
               throw new LocalizedException(__('Payment authorization transaction has been declined:  ' . $tran->error));
            } else {
            	$payment->setStatus(self::STATUS_ERROR);
               throw new LocalizedException(__('Payment authorization error:  ' . $tran->error . '('.$tran->errorcode . ')'));
            }
        } else {
            $payment->setStatus(self::STATUS_ERROR);
            throw new LocalizedException(__('Invalid transaction id '));
        }
        return $this;
    }

    protected function _initTransaction(InfoInterface $payment)
    {
        $tran = new TranApi();

        if ($this->getConfigData('sandbox'))
            $tran->usesandbox = true;



        switch ($this->getConfigData('currency')) {
            case "AFN":
                $tran->currency = '971';
                break;
            case "AWG":
                $tran->currency = '533';
                break;
            case "AUD":
                $tran->currency = '036';
                break;
            case "ARS":
                $tran->currency = '032';
                break;
            case "AZN":
                $tran->currency = '944';
                break;
            case "BSD":
                $tran->currency = '044';
                break;
            case "BDT":
                $tran->currency = '050';
                break;
            case "BBD":
                $tran->currency = '052';
                break;
            case "BYR":
                $tran->currency = '974';
                break;
            case "BOB":
                $tran->currency = '068';
                break;
            case "BRL":
                $tran->currency = '986';
                break;
            case "GBP":
                $tran->currency = '826';
                break;
            case "BGN":
                $tran->currency = '975';
                break;
            case "KHR":
                $tran->currency = '116';
                break;
            case "CAD":
                $tran->currency = '124';
                break;
            case "KYD":
                $tran->currency = '136';
                break;
            case "CLP":
                $tran->currency = '152';
                break;
            case "CNY":
                $tran->currency = '156';
                break;
            case "COP":
                $tran->currency = '170';
                break;
            case "CRC":
                $tran->currency = '188';
                break;
            case "HRK":
                $tran->currency = '191';
                break;
            case "CZK":
                $tran->currency = '203';
                break;
            case "DKK":
                $tran->currency = '208';
                break;
            case "DOP":
                $tran->currency = '214';
                break;
            case "XCD":
                $tran->currency = '951';
                break;
            case "EGP":
                $tran->currency = '818';
                break;
            case "ERN":
                $tran->currency = '232';
                break;
            case "EEK":
                $tran->currency = '233';
                break;
            case "EUR":
                $tran->currency = '978';
                break;
            case "GEL":
                $tran->currency = '981';
                break;
            case "GHS":
                $tran->currency = '288';
                break;
            case "GIP":
                $tran->currency = '292';
                break;
            case "GTQ":
                $tran->currency = '320';
                break;
            case "HNL":
                $tran->currency = '340';
                break;
            case "HKD":
                $tran->currency = '344';
                break;
            case "HUF":
                $tran->currency = '348';
                break;
            case "ISK":
                $tran->currency = '352';
                break;
            case "INR":
                $tran->currency = '356';
                break;
            case "IDR":
                $tran->currency = '360';
                break;
            case "ILS":
                $tran->currency = '376';
                break;
            case "JMD":
                $tran->currency = '388';
                break;
            case "JPY":
                $tran->currency = '392';
                break;
            case "KZT":
                $tran->currency = '368';
                break;
            case "KES":
                $tran->currency = '404';
                break;

            case "KWD":
                $tran->currency = '414';
                break;
            case "LVL":
                $tran->currency = '428';
                break;
            case "LBP":
                $tran->currency = '422';
                break;
            case "LTL":
                $tran->currency = '440';
                break;
            case "MOP":
                $tran->currency = '446';
                break;
            case "MKD":
                $tran->currency = '807';
                break;
            case "MGA":
                $tran->currency = '969';
                break;
            case "MYR":
                $tran->currency = '458';
                break;
            case "BAM":
                $tran->currency = '977';
                break;
            case "MUR":
                $tran->currency = '480';
                break;
            case "MXN":
                $tran->currency = '484';
                break;
            case "MZN":
                $tran->currency = '508';
                break;
            case "NPR":
                $tran->currency = '524';
                break;
            case "ANG":
                $tran->currency = '532';
                break;
            case "TWD":
                $tran->currency = '901';
                break;
            case "NZD":
                $tran->currency = '554';
                break;
            case "NIC":
                $tran->currency = '558';
                break;
            case "NGN":
                $tran->currency = '566';
                break;
            case "KPW":
                $tran->currency = '408';
                break;
            case "NOK":
                $tran->currency = '578';
                break;
            case "OMR":
                $tran->currency = '512';
                break;
            case "PKR":
                $tran->currency = '586';
                break;

            case "PYG":
                $tran->currency = '600';
                break;
            case "PEN":
                $tran->currency = '604';
                break;
            case "PHP":
                $tran->currency = '608';
                break;
            case "QAR":
                $tran->currency = '634';
                break;
            case "RON":
                $tran->currency = '946';
                break;
            case "RUB":
                $tran->currency = '643';
                break;
            case "SAR":
                $tran->currency = '682';
                break;
            case "RSD":
                $tran->currency = '891';
                break;
            case "SCR":
                $tran->currency = '690';
                break;
            case "SGD":
                $tran->currency = '702';
                break;
            case "SKK":
                $tran->currency = '703';
                break;
            case "ZAR":
                $tran->currency = '710';
                break;
            case "KRW":
                $tran->currency = '410';
                break;
            case "LKR":
                $tran->currency = '144';
                break;
            case "SRD":
                $tran->currency = '968';
                break;
            case "SEK":
                $tran->currency = '752';
                break;
            case "CHF":
                $tran->currency = '756';
                break;
            case "TZS":
                $tran->currency = '834';
                break;
            case "THB":
                $tran->currency = '764';
                break;
            case "TTD":
                $tran->currency = '780';
                break;
            case "TRY":
                $tran->currency = '949';
                break;
            case "AED":
                $tran->currency = '784';
                break;

            case "USD":
                $tran->currency = '840';
                break;
            case "UGX":
                $tran->currency = '800';
                break;
            case "UAH":
                $tran->currency = '980';
                break;
            case "UYU":
                $tran->currency = '858';
                break;
            case "UZS":
                $tran->currency = '860';
                break;
            case "VEB":
                $tran->currency = '862';
                break;
            case "VND":
                $tran->currency = '704';
                break;
            case "ZMK":
                $tran->currency = '894';
                break;
            case "ZWD":
                $tran->currency = '666';
                break;
            case "716":
                $tran->currency = '666';
                break;

            default:
                $tran->currency = '000';
        }

        $tran->key = $this->getConfigData('sourcekey');
        $tran->pin = $this->getConfigData('sourcepin');
        $tran->software = 'ZS_Usaepay 1.0.0';
        return $tran;
    }
	
	/**
     * @param int $cardId
     * @return Card
     */
    protected function loadCard($cardId)
    {
        /** @var Card $card */
        $card = $this->cardFactory->create();
        return $card->load($cardId);
    }
	
	/**
     * @throws LocalizedException
     */
    protected function saveCard($token,$type,$cardNo)
    {
        if ($this->loadedCardId) {
            return;
        }
        $info = $this->getInfoInstance();
        $saveCard = $info->getAdditionalInformation(self::SAVE_CARD);
		
        if (!$saveCard) {
            return;
        }
        if(!$this->configHelper->isTokenizationEnabled()) {
            return;
        }
        $customerId = $this->customerSession->getCustomerId();
        $data = [
            'cc_type' 			=> $type,
            'cc_last_4' 		=> $info->getData('cc_last_4'),
            'cc_exp_month' 		=> (int)$info->getData('cc_exp_month'),
            'cc_exp_year' 		=> (int)$info->getData('cc_exp_year'),
            'customer_id' 		=> $customerId,
            'cc_token' 			=>$token,
            'is_active' 		=> 1
        ];
        $card = $this->cardFactory->create();
        $card->addData($data);
        $card->save();

        $this->loadedCardId = $card->getId();
    }
}
